using System;
using System.Linq;
using TaleWorlds.CampaignSystem;
using TaleWorlds.CampaignSystem.Party;
using TaleWorlds.Core;
using TaleWorlds.Library;
using TaleWorlds.CampaignSystem.CharacterDevelopment;
using TaleWorlds.CampaignSystem.Extensions;

namespace MamaSaysImSPECIAL
{
    public class CharacterCustomizationBehavior : CampaignBehaviorBase
    {
        private bool _hasAppliedSettings = false;

        public override void RegisterEvents()
        {
            CampaignEvents.OnCharacterCreationIsOverEvent.AddNonSerializedListener(this, OnCharacterCreationIsOver);
        }

        public override void SyncData(IDataStore dataStore)
        {
            dataStore.SyncData("_hasAppliedSettings", ref _hasAppliedSettings);
        }

        // Phase 1: Execute immediately after character creation
        private void OnCharacterCreationIsOver()
        {
            if (_hasAppliedSettings)
                return;

            // Standalone config (no external settings dependency)
            var debug = MamaSaysImSPECIALConfig.EnableDebugMessages;

            var mainHero = Hero.MainHero;
            if (mainHero == null)
                return;

            try
            {
                // 1. S.P.E.C.I.A.L - Add Attribute Points
                if (MamaSaysImSPECIALConfig.AddAttributePoints)
                {
                    foreach (CharacterAttribute attribute in Attributes.All)
                    {
                        mainHero.HeroDeveloper.AddAttribute(attribute, 1, false);
                    }

                    if (debug)
                    {
                        InformationManager.DisplayMessage(new InformationMessage(
                            "[Mama Says I'm S.P.E.C.I.A.L.] Added 1 point to all attributes"));
                    }
                }

                // 2. Time To Focus - Add Focus Points
                if (MamaSaysImSPECIALConfig.AddFocusPoints)
                {
                    foreach (SkillObject skill in Skills.All)
                    {
                        mainHero.HeroDeveloper.AddFocus(skill, 1, false);
                    }

                    if (debug)
                    {
                        InformationManager.DisplayMessage(new InformationMessage(
                            "[Mama Says I'm S.P.E.C.I.A.L.] Added 1 focus point to all skills"));
                    }
                }

                // 3. LEVEL UP! - Set All Skills
                if (MamaSaysImSPECIALConfig.SetAllSkills && MamaSaysImSPECIALConfig.SkillLevel > 0)
                {
                    foreach (SkillObject skill in Skills.All)
                    {
                        mainHero.HeroDeveloper.ChangeSkillLevel(skill, MamaSaysImSPECIALConfig.SkillLevel, false);
                    }

                    if (debug)
                    {
                        InformationManager.DisplayMessage(new InformationMessage(
                            $"[Mama Says I'm S.P.E.C.I.A.L.] Set all skills to level {MamaSaysImSPECIALConfig.SkillLevel}"));
                    }
                }

                _hasAppliedSettings = true;
            }
            catch (Exception ex)
            {
                if (debug)
                {
                    InformationManager.DisplayMessage(new InformationMessage(
                        $"[Mama Says I'm S.P.E.C.I.A.L.] Error in Phase 1: {ex.Message}", Colors.Red));
                }
                _hasAppliedSettings = true; // Prevent retry loop
            }
        }


    }
}
